<?php

namespace App\Services;


class UrlHelperService
{
    /**
     * Extracts the base domain name from a given URL.
     *
     * This method provides a simplified approach suitable for typical domains
     * and development domains like '.test'. It does not fully handle all edge
     * cases of complex Public Suffix List TLDs (e.g., 'co.uk').
     *
     * @param string $url The URL from which to extract the base domain.
     * @return string|null The base domain name (e.g., "iranefe.test", "google.com"),
     * or null if the host cannot be determined or parsed.
     */
    public static function getBaseDomain(string $url): ?string
    {
        $host = parse_url($url, PHP_URL_HOST);

        if ($host === false || $host === null) {
            return null; // Could not parse host or host is missing
        }

        $host_parts = explode('.', $host);

        // Handle cases like 'localhost' or simple hostnames without a TLD
        if (count($host_parts) <= 1) {
            return $host;
        }

        // Return the last two parts joined by a dot.
        // This works for "example.com", "sub.example.com" -> "example.com"
        // and for "iranefe.test" -> "iranefe.test"
        return $host_parts[count($host_parts) - 2] . '.' . $host_parts[count($host_parts) - 1];
    }

    /**
     * A more robust (but still not perfect without a PSL) attempt to handle
     * potential two-part TLDs. This is more complex and less reliable
     * than a dedicated PSL library. For .test, the simpler approach is fine.
     *
     * @param string $url The URL.
     * @return string|null The base domain.
     * @deprecated This method is more complex and less reliable than using a dedicated PSL library.
     */
    public static function getBaseDomainAdvanced(string $url): ?string
    {
        $host = parse_url($url, PHP_URL_HOST);

        if ($host === false || $host === null) {
            return null;
        }

        $host_parts = explode('.', $host);
        $num_parts = count($host_parts);

        if ($num_parts <= 1) {
            return $host; // e.g., 'localhost'
        }

        // Basic heuristic for common two-part TLDs (like co.uk)
        // This is a very rough approximation and not foolproof!
        // A real solution needs a Public Suffix List.
        $last_part = $host_parts[$num_parts - 1];
        $second_last_part = $host_parts[$num_parts - 2];

        // If the last part is a common TLD (like com, org, net, info) and
        // the second to last part is also short (like co, gov, org),
        // we might assume it's a two-part TLD. This is highly simplified.
        $common_short_tlds = ['com', 'org', 'net', 'info', 'biz', 'gov', 'edu', 'mil', 'int', 'test'];
        $common_second_level_parts = ['co', 'gov', 'org', 'ac', 'ltd', 'plc', 'sch'];

        if (in_array($last_part, $common_short_tlds) && in_array($second_last_part, $common_second_level_parts)) {
            if ($num_parts >= 3) {
                return $host_parts[$num_parts - 3] . '.' . $second_last_part . '.' . $last_part;
            }
        }

        return $second_last_part . '.' . $last_part;
    }
}
