<?php

namespace App\Services\Models;

use App\Models\Setting;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;

class SettingService
{
    /**
     * the purpose of this variable is for to separate some option from other
     * for example if latter we add a flutter app specifically for team management
     * we create env named team-app or something
     *
     * accepted value web,api,app
     */
    private string $env;

    public Collection $settings;

    public function __construct($env = 'web')
    {
        $this->env = $env;
        $this->settings = Setting::env($this->env)->get();
    }

    public function env(string $env): static
    {
        // clone so chaining doesn't mutate global singleton instance
        $clone = clone $this;
        $clone->env = $env;
        $clone->settings = collect(); // force reload
        return $clone;
    }

    public function all(): mixed
    {
        $this->ensureLoaded();


        return $this->settings;
    }

    public function get(string $key, mixed $default = null): mixed
    {
        $this->ensureLoaded();

        $row = $this->settings->firstWhere('key', $key);

        return $row?->value ?? $default;
    }

    public function first(string $key): ?Setting
    {
        $this->ensureLoaded();
        return $this->settings->firstWhere('key', $key);
    }

    public function many(string $key): Collection
    {
        $this->ensureLoaded();
        return $this->settings->where('key', $key)->values();
    }

    public function group(string $group): Collection
    {
        $this->ensureLoaded();
        return $this->settings->where('group', $group)->values();
    }

    public function forgetCache(): void
    {
        Cache::forget("options:{$this->env}");
        $this->settings = collect();
    }

    protected function ensureLoaded(): void
    {
        if ($this->settings->isNotEmpty()) {
            return;
        }

        // Per-request cache (fast) + persistent cache (optional)
        $cacheKey = "options:{$this->env}";

        $this->settings = Cache::remember($cacheKey, now()->addMinutes(30), function () {
            return Setting::query()
                ->where('env', $this->env)
                ->get();
        });
    }

}
