<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use RegexIterator;

class FixSvgCurrentColor extends Command
{
    protected $signature = 'svg:current-color
        {--path=resources/svg/iconsax : Root folder to scan}
        {--dry-run : Show what would change without writing}
        {--only=* : Only process these attributes (stroke/fill), can pass multiple}
        {--ext=svg : File extension to process}';

    protected $description = 'Replace hard-coded SVG stroke/fill colors with currentColor (keeps fill="none").';

    public function handle(): int
    {
        $root = base_path($this->option('path'));
        $dryRun = (bool) $this->option('dry-run');

        $only = $this->option('only');
        $only = is_array($only) && count($only) ? array_map('strtolower', $only) : ['stroke', 'fill'];

        $ext = ltrim((string) $this->option('ext'), '.');

        if (!is_dir($root)) {
            $this->error("Path not found: {$root}");
            return self::FAILURE;
        }

        $files = $this->getFiles($root, $ext);

        $this->info("Scanning: {$root}");
        $this->info("Files: " . count($files));
        $this->info("Mode: " . ($dryRun ? 'DRY RUN' : 'WRITE'));
        $this->line('');

        $changedFiles = 0;
        $changedAttrs = 0;

        foreach ($files as $file) {
            $original = file_get_contents($file);
            if ($original === false) {
                $this->warn("Could not read: {$file}");
                continue;
            }

            $updated = $original;

            // Replace attribute values like stroke="#292D32" or stroke="rgb(...)" or stroke="black"
            // Keep stroke="none" and fill="none" unchanged.
            foreach ($only as $attr) {
                if (!in_array($attr, ['stroke', 'fill'], true)) {
                    continue;
                }

                // Match: stroke="..."/stroke='...'
                // Skip if value is none or currentColor or url(...) (gradients), or "transparent"
                $pattern = '/\b' . preg_quote($attr, '/') . '\s*=\s*(["\'])(.*?)\1/i';

                $updated = preg_replace_callback($pattern, function ($m) use ($attr, &$changedAttrs) {
                    $quote = $m[1];
                    $val = trim($m[2]);

                    $valLower = strtolower($val);

                    // Do not modify:
                    if ($valLower === 'none') return $m[0];
                    if ($valLower === 'currentcolor') return $m[0];
                    if ($valLower === 'transparent') return $m[0];
                    if (str_starts_with($valLower, 'url(')) return $m[0]; // gradients/patterns

                    // Only replace values that look like colors.
                    // Hex, rgb/rgba, hsl/hsla, or named colors.
                    $looksLikeColor =
                        preg_match('/^#([0-9a-f]{3}|[0-9a-f]{6}|[0-9a-f]{8})$/i', $val) ||
                        preg_match('/^(rgb|rgba|hsl|hsla)\(/i', $val) ||
                        preg_match('/^[a-z]+$/i', $val); // "black", "white", etc.

                    if (!$looksLikeColor) return $m[0];

                    $changedAttrs++;
                    return $attr . '=' . $quote . 'currentColor' . $quote;
                }, $updated) ?? $updated;
            }

            // Also handle inline style="stroke:#292D32; fill:#000"
            // Keep "none" and currentColor unchanged.
            $updated = preg_replace_callback('/\bstyle\s*=\s*(["\'])(.*?)\1/i', function ($m) use ($only, &$changedAttrs) {
                $quote = $m[1];
                $style = $m[2];

                $newStyle = $style;

                foreach ($only as $attr) {
                    if (!in_array($attr, ['stroke', 'fill'], true)) continue;

                    $newStyle = preg_replace_callback(
                        '/(\b' . preg_quote($attr, '/') . '\s*:\s*)([^;]+)(;?)/i',
                        function ($sm) use (&$changedAttrs) {
                            $prefix = $sm[1];
                            $val = trim($sm[2]);
                            $suffix = $sm[3] ?? '';

                            $valLower = strtolower($val);

                            if ($valLower === 'none') return $sm[0];
                            if ($valLower === 'currentcolor') return $sm[0];
                            if ($valLower === 'transparent') return $sm[0];
                            if (str_starts_with($valLower, 'url(')) return $sm[0];

                            $looksLikeColor =
                                preg_match('/^#([0-9a-f]{3}|[0-9a-f]{6}|[0-9a-f]{8})$/i', $val) ||
                                preg_match('/^(rgb|rgba|hsl|hsla)\(/i', $val) ||
                                preg_match('/^[a-z]+$/i', $val);

                            if (!$looksLikeColor) return $sm[0];

                            $changedAttrs++;
                            return $prefix . 'currentColor' . $suffix;
                        },
                        $newStyle
                    ) ?? $newStyle;
                }

                if ($newStyle === $style) return $m[0];
                return 'style=' . $quote . $newStyle . $quote;
            }, $updated) ?? $updated;

            if ($updated !== $original) {
                $changedFiles++;

                if ($dryRun) {
                    $this->line("Would update: {$file}");
                    continue;
                }

                $ok = file_put_contents($file, $updated);
                if ($ok === false) {
                    $this->warn("Could not write: {$file}");
                } else {
                    $this->line("Updated: {$file}");
                }
            }
        }

        $this->line('');
        $this->info("Changed files: {$changedFiles}");
        $this->info("Changed attributes: {$changedAttrs}");

        return self::SUCCESS;
    }

    /**
     * @return array<int, string>
     */
    private function getFiles(string $root, string $ext): array
    {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($root, RecursiveDirectoryIterator::SKIP_DOTS)
        );

        $regex = new RegexIterator($iterator, '/\.' . preg_quote($ext, '/') . '$/i');

        $files = [];
        foreach ($regex as $fileInfo) {
            $files[] = $fileInfo->getPathname();
        }

        sort($files);

        return $files;
    }
}
